package com.example.filestorage.controller;

import com.example.filestorage.entity.FileMetadata;
import com.example.filestorage.request.InitUploadRequest;
import com.example.filestorage.request.UploadChunkRequest;
import com.example.filestorage.response.BaseResponse;
import com.example.filestorage.response.InitUploadResponse;
import com.example.filestorage.service.FileDownloadService;
import com.example.filestorage.service.FileUploadService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.InputStreamResource;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.List;

@RestController
@RequestMapping("/api/file")
public class FileController {
    
    @Autowired
    private FileUploadService fileUploadService;
    
    @Autowired
    private FileDownloadService fileDownloadService;
    
    /**
     * 初始化分片上传
     */
    @PostMapping("/init-upload")
    public BaseResponse<InitUploadResponse> initUpload(@RequestBody InitUploadRequest request) {
        try {
            InitUploadResponse response = fileUploadService.initUpload(request);
            return BaseResponse.success(response);
        } catch (Exception e) {
            return BaseResponse.error("初始化上传失败: " + e.getMessage());
        }
    }
    
    /**
     * 上传文件分片
     */
    @PostMapping("/upload-chunk")
    public BaseResponse<String> uploadChunk(@RequestBody UploadChunkRequest request) {
        try {
            fileUploadService.uploadChunk(request);
            return BaseResponse.success("分片上传成功");
        } catch (Exception e) {
            return BaseResponse.error("分片上传失败: " + e.getMessage());
        }
    }
    
    /**
     * 完成分片上传
     */
    @PostMapping("/complete-upload/{uploadId}")
    public BaseResponse<FileMetadata> completeUpload(@PathVariable String uploadId) {
        try {
            FileMetadata fileMetadata = fileUploadService.completeUpload(uploadId);
            return BaseResponse.success(fileMetadata);
        } catch (Exception e) {
            return BaseResponse.error("完成上传失败: " + e.getMessage());
        }
    }
    
    /**
     * 检查分片是否存在（断点续传）
     */
    @GetMapping("/check-chunk/{uploadId}/{chunkIndex}")
    public BaseResponse<Boolean> checkChunkExists(@PathVariable String uploadId, @PathVariable int chunkIndex) {
        try {
            boolean exists = fileUploadService.checkChunkExists(uploadId, chunkIndex);
            return BaseResponse.success(exists);
        } catch (Exception e) {
            return BaseResponse.error("检查分片失败: " + e.getMessage());
        }
    }
    
    /**
     * 获取已上传的分片列表（断点续传）
     */
    @GetMapping("/uploaded-chunks/{uploadId}")
    public BaseResponse<List<Integer>> getUploadedChunks(@PathVariable String uploadId) {
        try {
            List<Integer> uploadedChunks = fileUploadService.getUploadedChunks(uploadId);
            return BaseResponse.success(uploadedChunks);
        } catch (Exception e) {
            return BaseResponse.error("获取已上传分片失败: " + e.getMessage());
        }
    }
    
    /**
     * 取消上传
     */
    @DeleteMapping("/cancel-upload/{uploadId}")
    public BaseResponse<String> cancelUpload(@PathVariable String uploadId) {
        try {
            fileUploadService.cancelUpload(uploadId);
            return BaseResponse.success("取消上传成功");
        } catch (Exception e) {
            return BaseResponse.error("取消上传失败: " + e.getMessage());
        }
    }
    
    /**
     * 下载文件
     */
    @GetMapping("/download/{fileMd5}")
    public void downloadFile(@PathVariable String fileMd5, HttpServletResponse response) {
        try {
            fileDownloadService.downloadFile(fileMd5, response);
        } catch (Exception e) {
            try {
                response.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
                response.getWriter().write("下载失败: " + e.getMessage());
            } catch (IOException ioException) {
                // 忽略
            }
        }
    }
    
    /**
     * 获取文件信息
     */
    @GetMapping("/info/{fileMd5}")
    public BaseResponse<FileMetadata> getFileInfo(@PathVariable String fileMd5) {
        try {
            FileMetadata fileMetadata = fileDownloadService.getFileMetadata(fileMd5);
            if (fileMetadata != null) {
                return BaseResponse.success(fileMetadata);
            } else {
                return BaseResponse.error("文件不存在");
            }
        } catch (Exception e) {
            return BaseResponse.error("获取文件信息失败: " + e.getMessage());
        }
    }
}