package com.example.filestorage.service.impl;

import com.example.filestorage.config.StorageProperties;
import com.example.filestorage.entity.FileMetadata;
import com.example.filestorage.repository.FileMetadataRepository;
import com.example.filestorage.service.FileDownloadService;
import com.example.filestorage.service.FileStorageService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.nio.file.Files;

@Service
public class FileDownloadServiceImpl implements FileDownloadService {
    
    @Autowired
    @Qualifier("minioFileStorageServiceImpl")
    private FileStorageService minioFileStorageService;
    
    @Autowired
    @Qualifier("ossFileStorageServiceImpl")
    private FileStorageService ossFileStorageService;
    
    @Autowired
    private FileMetadataRepository fileMetadataRepository;
    
    @Autowired
    private StorageProperties storageProperties;
    
    @Override
    public void downloadFile(String fileMd5, HttpServletResponse response) throws IOException {
        FileMetadata fileMetadata = getFileMetadata(fileMd5);
        if (fileMetadata == null) {
            throw new RuntimeException("文件不存在: " + fileMd5);
        }
        
        // 设置响应头
        response.setContentType(fileMetadata.getContentType());
        response.setHeader("Content-Disposition", "attachment; filename=\"" + fileMetadata.getOriginalName() + "\"");
        response.setContentLengthLong(fileMetadata.getFileSize());
        
        // 获取文件流并写入响应
        try (InputStream inputStream = getFileStream(fileMd5);
             OutputStream outputStream = response.getOutputStream()) {
            
            byte[] buffer = new byte[8192];
            int bytesRead;
            while ((bytesRead = inputStream.read(buffer)) != -1) {
                outputStream.write(buffer, 0, bytesRead);
            }
            outputStream.flush();
        }
    }
    
    @Override
    public InputStream getFileStream(String fileMd5) {
        FileMetadata fileMetadata = getFileMetadata(fileMd5);
        if (fileMetadata == null) {
            throw new RuntimeException("文件不存在: " + fileMd5);
        }
        
        FileStorageService storageService = getStorageService(fileMetadata.getStorageType());
        return storageService.downloadFile(fileMetadata.getBucketName(), fileMetadata.getObjectName());
    }
    
    @Override
    public FileMetadata getFileMetadata(String fileMd5) {
        return fileMetadataRepository.findByFileMd5(fileMd5);
    }
    
    private FileStorageService getStorageService(String storageType) {
        if ("OSS".equalsIgnoreCase(storageType)) {
            return ossFileStorageService;
        } else {
            return minioFileStorageService;
        }
    }
}