package com.example.filestorage.service.impl;

import com.example.filestorage.config.StorageProperties;
import com.example.filestorage.entity.FileMetadata;
import com.example.filestorage.entity.UploadChunk;
import com.example.filestorage.entity.UploadInfo;
import com.example.filestorage.repository.FileMetadataRepository;
import com.example.filestorage.repository.UploadChunkRepository;
import com.example.filestorage.repository.UploadInfoRepository;
import com.example.filestorage.service.FileStorageService;
import com.example.filestorage.service.FileUploadService;
import com.example.filestorage.request.InitUploadRequest;
import com.example.filestorage.request.UploadChunkRequest;
import com.example.filestorage.response.InitUploadResponse;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.security.MessageDigest;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.concurrent.ConcurrentHashMap;

@Service
public class FileUploadServiceImpl implements FileUploadService {
    
    @Autowired
    @Qualifier("minioFileStorageServiceImpl")
    private FileStorageService minioFileStorageService;
    
    @Autowired
    @Qualifier("ossFileStorageServiceImpl")
    private FileStorageService ossFileStorageService;
    
    @Autowired
    private StorageProperties storageProperties;
    
    @Autowired
    private FileMetadataRepository fileMetadataRepository;
    
    @Autowired
    private UploadInfoRepository uploadInfoRepository;
    
    @Autowired
    private UploadChunkRepository chunkRepository;
    
    // 使用内存缓存存储上传信息（实际生产环境应使用Redis）
    private final ConcurrentHashMap<String, UploadInfo> uploadCache = new ConcurrentHashMap<>();
    
    @Override
    @Transactional
    public InitUploadResponse initUpload(InitUploadRequest request) {
        try {
            // 生成上传ID
            String uploadId = UUID.randomUUID().toString();
            
            // 检查文件是否已存在（秒传功能）
            if (request.getFileMd5() != null && fileMetadataRepository.existsByFileMd5(request.getFileMd5())) {
                FileMetadata existingFile = fileMetadataRepository.findByFileMd5(request.getFileMd5());
                return new InitUploadResponse(uploadId, true, existingFile.getFileUrl(), "文件已存在，支持秒传");
            }
            
            // 创建上传信息记录
            UploadInfo uploadInfo = new UploadInfo();
            uploadInfo.setUploadId(uploadId);
            uploadInfo.setFileName(request.getFileName());
            uploadInfo.setFileSize(request.getFileSize());
            uploadInfo.setFileMd5(request.getFileMd5());
            uploadInfo.setTotalChunks(request.getTotalChunks());
            uploadInfo.setStorageType(request.getStorageType());
            uploadInfo.setStatus("init");
            
            uploadInfoRepository.save(uploadInfo);
            uploadCache.put(uploadId, uploadInfo);
            
            return new InitUploadResponse(uploadId, false, null, "初始化成功");
        } catch (Exception e) {
            throw new RuntimeException("初始化上传失败: " + e.getMessage(), e);
        }
    }
    
    @Override
    public void uploadChunk(UploadChunkRequest request) {
        try {
            // 保存分片到临时存储
            String chunkObjectName = "temp/" + request.getUploadId() + "/" + request.getChunkIndex();
            String bucketName = getBucketName(request.getStorageType());
            
            // 上传分片
            InputStream inputStream = new ByteArrayInputStream(request.getChunkData());
            FileMetadata chunkMetadata = getStorageService(request.getStorageType())
                .uploadFile(inputStream, request.getChunkData().length, "application/octet-stream", bucketName, chunkObjectName);
            
            // 记录分片信息
            UploadChunk chunk = new UploadChunk();
            chunk.setUploadId(request.getUploadId());
            chunk.setChunkIndex(request.getChunkIndex());
            chunk.setChunkSize(request.getChunkData().length);
            chunk.setObjectName(chunkObjectName);
            chunk.setUploaded(true);
            
            chunkRepository.save(chunk);
        } catch (Exception e) {
            throw new RuntimeException("上传分片失败: " + e.getMessage(), e);
        }
    }
    
    @Override
    @Transactional
    public FileMetadata completeUpload(String uploadId) {
        try {
            UploadInfo uploadInfo = uploadInfoRepository.findByUploadId(uploadId);
            if (uploadInfo == null) {
                throw new RuntimeException("上传信息不存在: " + uploadId);
            }
            
            // 验证所有分片是否已上传
            List<UploadChunk> chunks = chunkRepository.findByUploadId(uploadId);
            if (chunks.size() != uploadInfo.getTotalChunks()) {
                throw new RuntimeException("分片数量不匹配，无法合并文件");
            }
            
            // 按分片索引排序
            chunks.sort((a, b) -> Integer.compare(a.getChunkIndex(), b.getChunkIndex()));
            
            // 合并文件
            String finalObjectName = generateObjectName(uploadInfo.getFileName(), uploadInfo.getFileMd5());
            String bucketName = getBucketName(uploadInfo.getStorageType());
            
            // 这里需要实现分片合并逻辑，实际项目中可以使用对象存储的分片合并功能
            // 为了简化，这里将所有分片下载后合并再上传
            byte[] mergedData = mergeChunks(chunks);
            
            FileStorageService storageService = getStorageService(uploadInfo.getStorageType());
            InputStream inputStream = new ByteArrayInputStream(mergedData);
            FileMetadata fileMetadata = storageService.uploadFile(
                inputStream, 
                uploadInfo.getFileSize(), 
                getContentType(uploadInfo.getFileName()), 
                bucketName, 
                finalObjectName
            );
            
            // 保存文件元数据
            fileMetadata.setFileMd5(uploadInfo.getFileMd5());
            fileMetadata.setOriginalName(uploadInfo.getFileName());
            fileMetadata.setStorageType(uploadInfo.getStorageType());
            fileMetadata = fileMetadataRepository.save(fileMetadata);
            
            // 更新上传状态
            uploadInfo.setStatus("completed");
            uploadInfoRepository.save(uploadInfo);
            
            // 删除临时分片文件
            for (UploadChunk chunk : chunks) {
                storageService.deleteFile(bucketName, chunk.getObjectName());
            }
            chunkRepository.deleteByUploadId(uploadId);
            
            // 从缓存中移除
            uploadCache.remove(uploadId);
            
            return fileMetadata;
        } catch (Exception e) {
            throw new RuntimeException("合并文件失败: " + e.getMessage(), e);
        }
    }
    
    @Override
    public boolean checkChunkExists(String uploadId, int chunkIndex) {
        try {
            // 检查分片是否已存在
            return chunkRepository.existsByUploadIdAndChunkIndex(uploadId, chunkIndex);
        } catch (Exception e) {
            throw new RuntimeException("检查分片存在性失败: " + e.getMessage(), e);
        }
    }
    
    @Override
    public List<Integer> getUploadedChunks(String uploadId) {
        try {
            // 获取已上传的分片列表
            List<UploadChunk> chunks = chunkRepository.findByUploadId(uploadId);
            return chunks.stream()
                .map(UploadChunk::getChunkIndex)
                .sorted()
                .collect(Collectors.toList());
        } catch (Exception e) {
            throw new RuntimeException("获取已上传分片列表失败: " + e.getMessage(), e);
        }
    }
    
    @Override
    @Transactional
    public void cancelUpload(String uploadId) {
        try {
            UploadInfo uploadInfo = uploadInfoRepository.findByUploadId(uploadId);
            if (uploadInfo != null) {
                // 删除已上传的分片
                List<UploadChunk> chunks = chunkRepository.findByUploadId(uploadId);
                FileStorageService storageService = getStorageService(uploadInfo.getStorageType());
                String bucketName = getBucketName(uploadInfo.getStorageType());
                
                for (UploadChunk chunk : chunks) {
                    storageService.deleteFile(bucketName, chunk.getObjectName());
                }
                
                // 删除数据库记录
                chunkRepository.deleteByUploadId(uploadId);
                uploadInfoRepository.deleteByUploadId(uploadId);
                
                // 从缓存中移除
                uploadCache.remove(uploadId);
            }
        } catch (Exception e) {
            throw new RuntimeException("取消上传失败: " + e.getMessage(), e);
        }
    }
    
    private FileStorageService getStorageService(String storageType) {
        if ("OSS".equalsIgnoreCase(storageType)) {
            return ossFileStorageService;
        } else {
            return minioFileStorageService;
        }
    }
    
    private String getBucketName(String storageType) {
        if ("OSS".equalsIgnoreCase(storageType)) {
            return storageProperties.getOss().getBucketName();
        } else {
            return storageProperties.getMinio().getBucketName();
        }
    }
    
    private String generateObjectName(String fileName, String fileMd5) {
        // 生成唯一对象名，使用MD5前缀避免冲突
        String extension = getFileExtension(fileName);
        return "files/" + fileMd5.substring(0, 8) + "/" + fileMd5 + extension;
    }
    
    private String getFileExtension(String fileName) {
        int lastDotIndex = fileName.lastIndexOf('.');
        if (lastDotIndex > 0) {
            return fileName.substring(lastDotIndex);
        }
        return "";
    }
    
    private String getContentType(String fileName) {
        String extension = getFileExtension(fileName).toLowerCase();
        switch (extension) {
            case ".jpg":
            case ".jpeg":
                return "image/jpeg";
            case ".png":
                return "image/png";
            case ".gif":
                return "image/gif";
            case ".pdf":
                return "application/pdf";
            case ".txt":
                return "text/plain";
            case ".doc":
                return "application/msword";
            case ".docx":
                return "application/vnd.openxmlformats-officedocument.wordprocessingml.document";
            default:
                return "application/octet-stream";
        }
    }
    
    private byte[] mergeChunks(List<UploadChunk> chunks) throws Exception {
        // 从存储服务下载所有分片并合并
        // 实际项目中，应该使用对象存储的分片上传功能（如MinIO的composeObject或OSS的multipart upload）
        // 这里是简化实现，将所有分片下载到内存中合并
        
        // 按分片索引排序
        chunks.sort((a, b) -> Integer.compare(a.getChunkIndex(), b.getChunkIndex()));
        
        // 计算总大小
        int totalSize = chunks.stream().mapToInt(UploadChunk::getChunkSize).sum();
        byte[] result = new byte[totalSize];
        
        int offset = 0;
        for (UploadChunk chunk : chunks) {
            // 根据存储类型获取对应的服务
            FileStorageService storageService = getStorageService(getUploadInfo(chunk.getUploadId()).getStorageType());
            String bucketName = getBucketName(getUploadInfo(chunk.getUploadId()).getStorageType());
            
            // 下载分片数据
            try (java.io.InputStream chunkStream = storageService.downloadFile(bucketName, chunk.getObjectName())) {
                byte[] chunkData = chunkStream.readAllBytes();
                System.arraycopy(chunkData, 0, result, offset, chunkData.length);
                offset += chunkData.length;
            }
        }
        
        return result;
    }
    
    private UploadInfo getUploadInfo(String uploadId) {
        UploadInfo info = uploadCache.get(uploadId);
        if (info == null) {
            info = uploadInfoRepository.findByUploadId(uploadId);
            if (info != null) {
                uploadCache.put(uploadId, info);
            }
        }
        return info;
    }
}