package com.example.filestorage.service.impl;

import com.example.filestorage.entity.FileMetadata;
import com.example.filestorage.service.FileStorageService;
import io.minio.*;
import io.minio.errors.MinioException;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.io.InputStream;

@Service
public class MinioFileStorageServiceImpl implements FileStorageService {

    private final MinioClient minioClient;

    @Value("${minio.bucket-name}")
    private String defaultBucketName;

    public MinioFileStorageServiceImpl(MinioClient minioClient) {
        this.minioClient = minioClient;
    }

    @Override
    public FileMetadata uploadFile(MultipartFile file, String bucketName, String objectName) {
        try {
            if (bucketName == null || bucketName.isEmpty()) {
                bucketName = defaultBucketName;
            }

            // 检查存储桶是否存在，不存在则创建
            if (!minioClient.bucketExists(BucketExistsArgs.builder().bucket(bucketName).build())) {
                minioClient.makeBucket(MakeBucketArgs.builder().bucket(bucketName).build());
            }

            // 上传文件
            minioClient.putObject(
                PutObjectArgs.builder()
                    .bucket(bucketName)
                    .object(objectName)
                    .stream(file.getInputStream(), file.getSize(), -1)
                    .contentType(file.getContentType())
                    .build()
            );

            // 创建文件元数据
            FileMetadata metadata = new FileMetadata();
            metadata.setBucketName(bucketName);
            metadata.setObjectName(objectName);
            metadata.setFileName(file.getOriginalFilename());
            metadata.setFileSize(file.getSize());
            metadata.setContentType(file.getContentType());
            metadata.setStorageType("MINIO");
            metadata.setUrl(getFileUrl(bucketName, objectName));

            return metadata;
        } catch (Exception e) {
            throw new RuntimeException("上传文件失败", e);
        }
    }

    @Override
    public FileMetadata uploadFile(InputStream inputStream, long size, String contentType, String bucketName, String objectName) {
        try {
            if (bucketName == null || bucketName.isEmpty()) {
                bucketName = defaultBucketName;
            }

            // 检查存储桶是否存在，不存在则创建
            if (!minioClient.bucketExists(BucketExistsArgs.builder().bucket(bucketName).build())) {
                minioClient.makeBucket(MakeBucketArgs.builder().bucket(bucketName).build());
            }

            // 上传文件
            minioClient.putObject(
                PutObjectArgs.builder()
                    .bucket(bucketName)
                    .object(objectName)
                    .stream(inputStream, size, -1)
                    .contentType(contentType)
                    .build()
            );

            // 创建文件元数据
            FileMetadata metadata = new FileMetadata();
            metadata.setBucketName(bucketName);
            metadata.setObjectName(objectName);
            metadata.setFileSize(size);
            metadata.setContentType(contentType);
            metadata.setStorageType("MINIO");
            metadata.setUrl(getFileUrl(bucketName, objectName));

            return metadata;
        } catch (Exception e) {
            throw new RuntimeException("上传文件失败", e);
        }
    }

    @Override
    public InputStream downloadFile(String bucketName, String objectName) {
        try {
            if (bucketName == null || bucketName.isEmpty()) {
                bucketName = defaultBucketName;
            }

            return minioClient.getObject(
                GetObjectArgs.builder()
                    .bucket(bucketName)
                    .object(objectName)
                    .build()
            );
        } catch (Exception e) {
            throw new RuntimeException("下载文件失败", e);
        }
    }

    @Override
    public void deleteFile(String bucketName, String objectName) {
        try {
            if (bucketName == null || bucketName.isEmpty()) {
                bucketName = defaultBucketName;
            }

            minioClient.removeObject(
                RemoveObjectArgs.builder()
                    .bucket(bucketName)
                    .object(objectName)
                    .build()
            );
        } catch (Exception e) {
            throw new RuntimeException("删除文件失败", e);
        }
    }

    @Override
    public boolean exists(String bucketName, String objectName) {
        try {
            if (bucketName == null || bucketName.isEmpty()) {
                bucketName = defaultBucketName;
            }

            minioClient.statObject(
                StatObjectArgs.builder()
                    .bucket(bucketName)
                    .object(objectName)
                    .build()
            );
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    @Override
    public String getFileUrl(String bucketName, String objectName) {
        if (bucketName == null || bucketName.isEmpty()) {
            bucketName = defaultBucketName;
        }
        // 返回预签名URL，这里可以根据需要设置过期时间
        try {
            return minioClient.getPresignedObjectUrl(
                GetPresignedObjectUrlArgs.builder()
                    .method(Method.GET)
                    .bucket(bucketName)
                    .object(objectName)
                    .expiry(7, java.util.concurrent.TimeUnit.DAYS) // 7天有效期
                    .build()
            );
        } catch (Exception e) {
            throw new RuntimeException("生成文件访问URL失败", e);
        }
    }
}