package com.example.filestorage.service.impl;

import com.aliyun.oss.OSS;
import com.aliyun.oss.model.GetObjectRequest;
import com.aliyun.oss.model.PutObjectRequest;
import com.aliyun.oss.model.PutObjectResult;
import com.example.filestorage.config.StorageProperties;
import com.example.filestorage.entity.FileMetadata;
import com.example.filestorage.service.FileStorageService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.io.ByteArrayInputStream;
import java.io.InputStream;

@Service
public class OSSFileStorageServiceImpl implements FileStorageService {
    
    @Autowired
    private OSS ossClient;
    
    @Autowired
    private StorageProperties storageProperties;
    
    @Override
    public FileMetadata uploadFile(MultipartFile file, String bucketName, String objectName) {
        try {
            // 上传文件到OSS
            PutObjectRequest putObjectRequest = new PutObjectRequest(
                storageProperties.getOss().getBucketName(), 
                objectName, 
                file.getInputStream()
            );
            PutObjectResult result = ossClient.putObject(putObjectRequest);
            
            // 创建文件元数据
            FileMetadata metadata = new FileMetadata();
            metadata.setBucketName(bucketName);
            metadata.setObjectName(objectName);
            metadata.setOriginalName(file.getOriginalFilename());
            metadata.setFileSize(file.getSize());
            metadata.setContentType(file.getContentType());
            metadata.setStorageType("OSS");
            
            return metadata;
        } catch (Exception e) {
            throw new RuntimeException("OSS文件上传失败: " + e.getMessage(), e);
        }
    }
    
    @Override
    public FileMetadata uploadFile(InputStream inputStream, long size, String contentType, String bucketName, String objectName) {
        try {
            // 上传文件到OSS
            PutObjectRequest putObjectRequest = new PutObjectRequest(
                storageProperties.getOss().getBucketName(), 
                objectName, 
                inputStream
            );
            PutObjectResult result = ossClient.putObject(putObjectRequest);
            
            // 创建文件元数据
            FileMetadata metadata = new FileMetadata();
            metadata.setBucketName(bucketName);
            metadata.setObjectName(objectName);
            metadata.setFileSize(size);
            metadata.setContentType(contentType);
            metadata.setStorageType("OSS");
            
            return metadata;
        } catch (Exception e) {
            throw new RuntimeException("OSS文件上传失败: " + e.getMessage(), e);
        }
    }
    
    @Override
    public InputStream downloadFile(String bucketName, String objectName) {
        try {
            // 从OSS下载文件
            GetObjectRequest getObjectRequest = new GetObjectRequest(
                storageProperties.getOss().getBucketName(), 
                objectName
            );
            return ossClient.getObject(getObjectRequest).getObjectContent();
        } catch (Exception e) {
            throw new RuntimeException("OSS文件下载失败: " + e.getMessage(), e);
        }
    }
    
    @Override
    public void deleteFile(String bucketName, String objectName) {
        try {
            // 删除OSS文件
            ossClient.deleteObject(storageProperties.getOss().getBucketName(), objectName);
        } catch (Exception e) {
            throw new RuntimeException("OSS文件删除失败: " + e.getMessage(), e);
        }
    }
    
    @Override
    public boolean exists(String bucketName, String objectName) {
        try {
            return ossClient.doesObjectExist(storageProperties.getOss().getBucketName(), objectName);
        } catch (Exception e) {
            throw new RuntimeException("检查OSS文件存在性失败: " + e.getMessage(), e);
        }
    }
    
    @Override
    public String getFileUrl(String bucketName, String objectName) {
        // 生成OSS文件访问URL
        return "https://" + storageProperties.getOss().getBucketName() + "." + 
               storageProperties.getOss().getEndpoint() + "/" + objectName;
    }
}
